<?php
/**
 * API Integration Tests
 * Tests the actual API endpoints with real sample files
 */

require_once __DIR__ . '/../src/classes/autoload.php';

use Redact\Classes\PIIDetectionService;

class APIIntegrationTest
{
    private array $results = [];
    private string $samplesDir;
    private string $apiBaseUrl;
    
    public function __construct()
    {
        $this->samplesDir = __DIR__ . '/Samples';
        $this->apiBaseUrl = 'http://localhost/redact/api/v1';
    }
    
    public function run(): void
    {
        echo "APIIntegrationTest:\n";
        
        $this->testServiceDirectProcessing();
        $this->testServiceReturnsCorrectPageCount();
        $this->testServiceReturnsPageArray();
        $this->testServicePageArrayMatchesCount();
        $this->testAPIEndpointReachable();
        
        $this->displayResults();
    }
    
    private function testServiceDirectProcessing(): void
    {
        // Test PIIDetectionService directly (bypasses HTTP API)
        $sampleFile = $this->samplesDir . '/UPDATED Regulated Employment Reference.pdf';
        
        if (!file_exists($sampleFile)) {
            $this->skip('Service processes sample file correctly (file not found)');
            return;
        }
        
        // Check if config exists
        $configFile = __DIR__ . '/../src/config/config.php';
        if (!file_exists($configFile)) {
            $this->skip('Service processes sample file correctly (config not found)');
            return;
        }
        
        try {
            // Load config
            require_once $configFile;
            
            $awsCredentials = getConfig('AWS_Credentials');
            if (!$awsCredentials || !isset($awsCredentials['key']) || !isset($awsCredentials['secret'])) {
                $this->skip('Service processes sample file correctly (AWS credentials not configured)');
                return;
            }
            
            // Create service
            $service = new PIIDetectionService($awsCredentials, [
                'region' => 'us-east-1',
                'max_file_size' => 10 * 1024 * 1024
            ]);
            
            // Create thread
            $threadResult = $service->createThread(['test' => 'api_integration']);
            if (!$threadResult['success']) {
                $this->skip('Service processes sample file correctly (thread creation failed)');
                return;
            }
            
            $threadId = $threadResult['thread_id'];
            
            // Process file directly
            $result = $service->processDocument($sampleFile, $threadId);
            
            // Clean up thread
            $service->getThreadManager()->deleteThread($threadId);
            
            $this->assert(
                $result['success'] === true,
                'PIIDetectionService processes sample file successfully'
            );
            
        } catch (Exception $e) {
            $this->skip('Service processes sample file correctly (' . $e->getMessage() . ')');
        }
    }
    
    private function testServiceReturnsCorrectPageCount(): void
    {
        $sampleFile = $this->samplesDir . '/UPDATED Regulated Employment Reference.pdf';
        
        if (!file_exists($sampleFile)) {
            $this->skip('Service returns correct page count (file not found)');
            return;
        }
        
        $configFile = __DIR__ . '/../src/config/config.php';
        if (!file_exists($configFile)) {
            $this->skip('Service returns correct page count (config not found)');
            return;
        }
        
        try {
            require_once $configFile;
            
            $awsCredentials = getConfig('AWS_Credentials');
            if (!$awsCredentials || !isset($awsCredentials['key']) || !isset($awsCredentials['secret'])) {
                $this->skip('Service returns correct page count (AWS credentials not configured)');
                return;
            }
            
            $service = new PIIDetectionService($awsCredentials, [
                'region' => 'us-east-1'
            ]);
            
            $threadResult = $service->createThread(['test' => 'page_count']);
            $threadId = $threadResult['thread_id'];
            
            $result = $service->processDocument($sampleFile, $threadId);
            
            $service->getThreadManager()->deleteThread($threadId);
            
            if (!$result['success']) {
                $this->skip('Service returns correct page count (processing failed)');
                return;
            }
            
            // The sample file has 6 pages
            $expectedPageCount = 6;
            $actualPageCount = $result['total_pages'] ?? 0;
            
            $this->assert(
                isset($result['total_pages']),
                'Service response includes total_pages field'
            );
            
            $this->assert(
                $actualPageCount === $expectedPageCount,
                "Service returns correct page count (expected: $expectedPageCount, actual: $actualPageCount)"
            );
            
        } catch (Exception $e) {
            $this->skip('Service returns correct page count (' . $e->getMessage() . ')');
        }
    }
    
    private function testServiceReturnsPageArray(): void
    {
        $sampleFile = $this->samplesDir . '/UPDATED Regulated Employment Reference.pdf';
        
        if (!file_exists($sampleFile)) {
            $this->skip('Service returns pages array (file not found)');
            return;
        }
        
        $configFile = __DIR__ . '/../src/config/config.php';
        if (!file_exists($configFile)) {
            $this->skip('Service returns pages array (config not found)');
            return;
        }
        
        try {
            require_once $configFile;
            
            $awsCredentials = getConfig('AWS_Credentials');
            if (!$awsCredentials || !isset($awsCredentials['key']) || !isset($awsCredentials['secret'])) {
                $this->skip('Service returns pages array (AWS credentials not configured)');
                return;
            }
            
            $service = new PIIDetectionService($awsCredentials, [
                'region' => 'us-east-1'
            ]);
            
            $threadResult = $service->createThread(['test' => 'pages_array']);
            $threadId = $threadResult['thread_id'];
            
            $result = $service->processDocument($sampleFile, $threadId);
            
            $service->getThreadManager()->deleteThread($threadId);
            
            if (!$result['success']) {
                $this->skip('Service returns pages array (processing failed)');
                return;
            }
            
            $this->assert(
                isset($result['pages']) && is_array($result['pages']),
                'Service response includes pages array'
            );
            
        } catch (Exception $e) {
            $this->skip('Service returns pages array (' . $e->getMessage() . ')');
        }
    }
    
    private function testServicePageArrayMatchesCount(): void
    {
        $sampleFile = $this->samplesDir . '/UPDATED Regulated Employment Reference.pdf';
        
        if (!file_exists($sampleFile)) {
            $this->skip('Pages array length matches total_pages (file not found)');
            return;
        }
        
        $configFile = __DIR__ . '/../src/config/config.php';
        if (!file_exists($configFile)) {
            $this->skip('Pages array length matches total_pages (config not found)');
            return;
        }
        
        try {
            require_once $configFile;
            
            $awsCredentials = getConfig('AWS_Credentials');
            if (!$awsCredentials || !isset($awsCredentials['key']) || !isset($awsCredentials['secret'])) {
                $this->skip('Pages array length matches total_pages (AWS credentials not configured)');
                return;
            }
            
            $service = new PIIDetectionService($awsCredentials, [
                'region' => 'us-east-1'
            ]);
            
            $threadResult = $service->createThread(['test' => 'array_match']);
            $threadId = $threadResult['thread_id'];
            
            $result = $service->processDocument($sampleFile, $threadId);
            
            $service->getThreadManager()->deleteThread($threadId);
            
            if (!$result['success']) {
                $this->skip('Pages array length matches total_pages (processing failed)');
                return;
            }
            
            $totalPages = $result['total_pages'] ?? 0;
            $pagesArrayLength = count($result['pages'] ?? []);
            
            $this->assert(
                $pagesArrayLength === $totalPages,
                "Pages array length matches total_pages ($pagesArrayLength pages in array = $totalPages total_pages)"
            );
            
            // Verify each page has correct page_number
            if (isset($result['pages']) && is_array($result['pages'])) {
                $allPagesNumbered = true;
                foreach ($result['pages'] as $index => $page) {
                    if (!isset($page['page_number']) || $page['page_number'] !== ($index + 1)) {
                        $allPagesNumbered = false;
                        break;
                    }
                }
                
                $this->assert(
                    $allPagesNumbered,
                    'Each page in array has correct sequential page_number (1, 2, 3, ...)'
                );
            }
            
        } catch (Exception $e) {
            $this->skip('Pages array length matches total_pages (' . $e->getMessage() . ')');
        }
    }
    
    private function testAPIEndpointReachable(): void
    {
        // Simple connectivity test
        $threadCreateUrl = $this->apiBaseUrl . '/thread_create.php';
        
        $ch = curl_init($threadCreateUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode(['metadata' => ['test' => 'connectivity']]));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 0) {
            $this->skip('API endpoint is reachable (server not responding)');
            return;
        }
        
        $this->assert(
            $httpCode === 200 || $httpCode === 201,
            "API endpoint is reachable and responding (HTTP $httpCode)"
        );
    }
    
    private function assert(bool $condition, string $message, string $error = ''): void
    {
        $this->results[] = [
            'passed' => $condition,
            'message' => $message,
            'error' => $error
        ];
        
        $status = $condition ? '✓' : '✗';
        $output = "  $status $message";
        if (!$condition && $error) {
            $output .= " - Error: $error";
        }
        echo "$output\n";
    }
    
    private function skip(string $message): void
    {
        $this->results[] = [
            'passed' => true,
            'message' => $message,
            'skipped' => true
        ];
        echo "  ⊘ $message (skipped)\n";
    }
    
    private function displayResults(): void
    {
        $passed = count(array_filter($this->results, fn($r) => $r['passed']));
        $total = count($this->results);
        echo "\n";
    }
}

// Run tests if executed directly
if (basename(__FILE__) === basename($_SERVER['PHP_SELF'])) {
    $test = new APIIntegrationTest();
    $test->run();
}
