<?php
/**
 * Unit Tests for PIIDetectionService
 */

require_once __DIR__ . '/../src/classes/autoload.php';

use Redact\Classes\PIIDetectionService;

class PIIDetectionServiceTest
{
    private array $results = [];
    private string $testDataDir;
    
    public function __construct()
    {
        $this->testDataDir = __DIR__ . '/test_data/pii_service';
        if (!is_dir($this->testDataDir)) {
            mkdir($this->testDataDir, 0755, true);
        }
    }
    
    public function run(): void
    {
        echo "PIIDetectionServiceTest:\n";
        
        $this->testConstructor();
        $this->testConstructorWithConfig();
        $this->testCreateThread();
        $this->testGetThreadManager();
        $this->testProcessDocumentInvalidThread();
        $this->testProcessDocumentFileNotFound();
        $this->testProcessDocumentInvalidFile();
        
        $this->displayResults();
    }
    
    private function testConstructor(): void
    {
        try {
            $service = new PIIDetectionService([
                'key' => 'test_key',
                'secret' => 'test_secret'
            ]);
            $this->assert(true, 'Constructor with minimal credentials');
        } catch (Exception $e) {
            $this->assert(false, 'Constructor with minimal credentials', $e->getMessage());
        }
    }
    
    private function testConstructorWithConfig(): void
    {
        try {
            $service = new PIIDetectionService(
                [
                    'key' => 'test_key',
                    'secret' => 'test_secret'
                ],
                [
                    'region' => 'us-west-2',
                    'max_file_size' => 10485760,
                    'data_dir' => $this->testDataDir,
                    'thread_expiry_days' => 60
                ]
            );
            $this->assert(true, 'Constructor with full configuration');
        } catch (Exception $e) {
            $this->assert(false, 'Constructor with full configuration', $e->getMessage());
        }
    }
    
    private function testCreateThread(): void
    {
        $service = new PIIDetectionService([
            'key' => 'test_key',
            'secret' => 'test_secret'
        ], [
            'data_dir' => $this->testDataDir
        ]);
        
        $result = $service->createThread(['user_id' => 'test_123']);
        
        $this->assert(
            $result['success'] === true && isset($result['thread_id']),
            'createThread creates new thread via service'
        );
        
        // Cleanup
        if (isset($result['thread_id'])) {
            $service->getThreadManager()->deleteThread($result['thread_id']);
        }
    }
    
    private function testGetThreadManager(): void
    {
        $service = new PIIDetectionService([
            'key' => 'test_key',
            'secret' => 'test_secret'
        ], [
            'data_dir' => $this->testDataDir
        ]);
        
        $manager = $service->getThreadManager();
        
        $this->assert(
            $manager instanceof \Redact\Classes\ThreadManager,
            'getThreadManager returns ThreadManager instance'
        );
    }
    
    private function testProcessDocumentInvalidThread(): void
    {
        $service = new PIIDetectionService([
            'key' => 'test_key',
            'secret' => 'test_secret'
        ], [
            'data_dir' => $this->testDataDir
        ]);
        
        $result = $service->processDocument('/fake/path.pdf', 'thread_invalid_123');
        
        $this->assert(
            $result['success'] === false && strpos($result['error'], 'thread') !== false,
            'processDocument rejects invalid thread ID'
        );
    }
    
    private function testProcessDocumentFileNotFound(): void
    {
        $service = new PIIDetectionService([
            'key' => 'test_key',
            'secret' => 'test_secret'
        ], [
            'data_dir' => $this->testDataDir
        ]);
        
        // Create valid thread
        $threadResult = $service->createThread();
        $threadId = $threadResult['thread_id'];
        
        $result = $service->processDocument('/nonexistent/file.pdf', $threadId);
        
        $this->assert(
            $result['success'] === false && strpos($result['error'], 'not found') !== false,
            'processDocument rejects non-existent file'
        );
        
        // Cleanup
        $service->getThreadManager()->deleteThread($threadId);
    }
    
    private function testProcessDocumentInvalidFile(): void
    {
        $service = new PIIDetectionService([
            'key' => 'test_key',
            'secret' => 'test_secret'
        ], [
            'data_dir' => $this->testDataDir,
            'max_file_size' => 1024 // 1KB
        ]);
        
        // Create thread
        $threadResult = $service->createThread();
        $threadId = $threadResult['thread_id'];
        
        // Create file that's too large
        $testFile = $this->testDataDir . '/large_file.txt';
        file_put_contents($testFile, str_repeat('X', 2048)); // 2KB
        
        $result = $service->processDocument($testFile, $threadId);
        
        $this->assert(
            $result['success'] === false && strpos($result['error'], 'exceeds') !== false,
            'processDocument rejects file exceeding size limit'
        );
        
        // Cleanup
        unlink($testFile);
        $service->getThreadManager()->deleteThread($threadId);
    }
    
    private function assert(bool $condition, string $message, string $error = ''): void
    {
        $this->results[] = [
            'passed' => $condition,
            'message' => $message,
            'error' => $error
        ];
        
        $status = $condition ? '✓' : '✗';
        $output = "  $status $message";
        if (!$condition && $error) {
            $output .= " - Error: $error";
        }
        echo "$output\n";
    }
    
    private function displayResults(): void
    {
        $passed = count(array_filter($this->results, fn($r) => $r['passed']));
        $total = count($this->results);
        echo "\n";
    }
    
    public function __destruct()
    {
        // Cleanup test data
        $this->cleanupDirectory($this->testDataDir);
    }
    
    private function cleanupDirectory(string $dir): void
    {
        if (!is_dir($dir)) {
            return;
        }
        
        $items = array_diff(scandir($dir), ['.', '..']);
        foreach ($items as $item) {
            $path = $dir . '/' . $item;
            if (is_dir($path)) {
                $this->cleanupDirectory($path);
            } else {
                unlink($path);
            }
        }
        rmdir($dir);
    }
}

// Run tests if executed directly
if (basename(__FILE__) === basename($_SERVER['PHP_SELF'])) {
    $test = new PIIDetectionServiceTest();
    $test->run();
}
