<?php
/**
 * Test Runner
 * Executes all unit tests and displays results
 */

// Set error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Define base paths
define('BASE_PATH', dirname(__DIR__));
define('TEST_PATH', __DIR__);

// Load autoloader
require_once BASE_PATH . '/src/classes/autoload.php';

class TestRunner
{
    private array $results = [];
    private int $totalTests = 0;
    private int $passedTests = 0;
    private int $failedTests = 0;
    private int $skippedTests = 0;
    
    public function run(): void
    {
        echo "\n";
        echo "===========================================\n";
        echo "  REDACT API - UNIT TEST RUNNER\n";
        echo "===========================================\n\n";
        
        $startTime = microtime(true);
        
        // Run all test files
        $testFiles = [
            'ComprehendServiceTest.php',
            'FileHandlerTest.php',
            'TextractServiceTest.php',
            'ThreadManagerTest.php',
            'RegistryManagerTest.php',
            'PIIDetectionServiceTest.php',
            'SampleFilesTest.php',
            'APIIntegrationTest.php',
            'Utf8SanitizationTest.php'
        ];
        
        foreach ($testFiles as $testFile) {
            $this->runTestFile($testFile);
        }
        
        $endTime = microtime(true);
        $duration = round(($endTime - $startTime) * 1000, 2);
        
        // Display results
        $this->displayResults($duration);
    }
    
    private function runTestFile(string $filename): void
    {
        $filepath = TEST_PATH . '/' . $filename;
        
        if (!file_exists($filepath)) {
            echo "⚠ Test file not found: $filename\n";
            return;
        }
        
        $className = str_replace('.php', '', $filename);
        echo "Running $className...\n";
        
        try {
            // Include the test file
            require_once $filepath;
            
            // Instantiate and run the test class
            if (class_exists($className)) {
                ob_start();
                $testInstance = new $className();
                $testInstance->run();
                $output = ob_get_clean();
                
                // Parse output for results
                $this->parseTestOutput($className, $output);
            } else {
                echo "✗ Error: Class $className not found\n";
                $this->results[$className] = [
                    'status' => 'error',
                    'message' => "Class not found",
                    'passed' => 0,
                    'failed' => 1,
                    'skipped' => 0,
                    'total' => 1
                ];
            }
            
        } catch (Exception $e) {
            echo "✗ Error running $className: " . $e->getMessage() . "\n";
            $this->results[$className] = [
                'status' => 'error',
                'message' => $e->getMessage(),
                'passed' => 0,
                'failed' => 1,
                'skipped' => 0,
                'total' => 1
            ];
        }
    }
    
    private function parseTestOutput(string $className, string $output): void
    {
        // Count test results from output
        $passed = substr_count($output, '✓');
        $failed = substr_count($output, '✗');
        $skipped = substr_count($output, '⊘');
        $total = $passed + $failed + $skipped;
        
        $this->totalTests += $total;
        $this->passedTests += $passed;
        $this->failedTests += $failed;
        $this->skippedTests += $skipped;
        
        $this->results[$className] = [
            'status' => $failed > 0 ? 'failed' : 'passed',
            'passed' => $passed,
            'failed' => $failed,
            'skipped' => $skipped,
            'total' => $total,
            'output' => $output
        ];
        
        // Display class results
        $status = $failed > 0 ? '✗ FAILED' : '✓ PASSED';
        $color = $failed > 0 ? '' : '';
        echo "  $status ($passed/$total tests)\n";
        
        // Show failures
        if ($failed > 0) {
            echo "  Details:\n";
            $lines = explode("\n", $output);
            foreach ($lines as $line) {
                if (strpos($line, '✗') !== false) {
                    echo "    " . trim($line) . "\n";
                }
            }
        }
        
        echo "\n";
    }
    
    private function displayResults(float $duration): void
    {
        echo "===========================================\n";
        echo "  TEST RESULTS SUMMARY\n";
        echo "===========================================\n\n";
        
        foreach ($this->results as $className => $result) {
            $status = $result['status'] === 'passed' ? '✓' : '✗';
            $statusText = $result['status'] === 'passed' ? 'PASSED' : 'FAILED';
            
            printf("%-40s %s %s (%d/%d)\n", 
                $className . ':', 
                $status, 
                $statusText,
                $result['passed'],
                $result['total']
            );
        }
        
        echo "\n===========================================\n";
        echo "  FINAL RESULTS\n";
        echo "===========================================\n\n";
        $actualTests = $this->passedTests + $this->failedTests; // Don't count skipped in total run
        echo "Total Tests:    $actualTests\n";
        echo "✓ Passed:       $this->passedTests\n";
        
        if ($this->failedTests > 0) {
            echo "✗ Failed:       $this->failedTests\n";
        }
        
        if ($this->skippedTests > 0) {
            echo "⊘ Skipped:      $this->skippedTests (optional tests)\n";
        }
        
        echo "Duration:       {$duration}ms\n";
        echo "\n";
        
        if ($this->failedTests === 0) {
            echo "🎉 ALL TESTS PASSED! 🎉\n\n";
            exit(0);
        } else {
            echo "❌ SOME TESTS FAILED\n\n";
            exit(1);
        }
    }
}

// Run tests
$runner = new TestRunner();
$runner->run();
