<?php
/**
 * Unit Tests for TextractService
 */

require_once __DIR__ . '/../src/classes/autoload.php';

use Redact\Classes\TextractService;

class TextractServiceTest
{
    private array $results = [];
    
    public function run(): void
    {
        echo "TextractServiceTest:\n";
        
        $this->testConstructor();
        $this->testConstructorWithRegion();
        $this->testAnalyzeDocumentStructure();
        $this->testAnalyzeDocumentFullStructure();
        $this->testParseTablesEmptyResponse();
        $this->testParseTablesWithData();
        
        $this->displayResults();
    }
    
    private function testConstructor(): void
    {
        try {
            $service = new TextractService('test_key', 'test_secret');
            $this->assert(true, 'Constructor with default region');
        } catch (Exception $e) {
            $this->assert(false, 'Constructor with default region', $e->getMessage());
        }
    }
    
    private function testConstructorWithRegion(): void
    {
        try {
            $service = new TextractService('test_key', 'test_secret', 'eu-west-1');
            $this->assert(true, 'Constructor with custom region');
        } catch (Exception $e) {
            $this->assert(false, 'Constructor with custom region', $e->getMessage());
        }
    }
    
    private function testAnalyzeDocumentStructure(): void
    {
        // Skip if GD is not available
        if (!extension_loaded('gd')) {
            $this->skip('analyzeDocument returns structured response (requires GD)');
            $this->skip('analyzeDocument returns false with invalid credentials (requires GD)');
            return;
        }
        
        $service = new TextractService('test_key', 'test_secret');
        
        // Create minimal test image data
        $image = imagecreate(100, 100);
        imagecolorallocate($image, 255, 255, 255);
        ob_start();
        imagejpeg($image);
        $imageData = ob_get_clean();
        imagedestroy($image);
        
        $result = $service->analyzeDocument($imageData);
        
        $this->assert(
            is_array($result) && isset($result['success']),
            'analyzeDocument returns structured response'
        );
        
        // With invalid credentials, should fail
        $this->assert(
            $result['success'] === false,
            'analyzeDocument returns false with invalid credentials'
        );
    }
    
    private function testAnalyzeDocumentFullStructure(): void
    {
        // Skip if GD is not available
        if (!extension_loaded('gd')) {
            $this->skip('analyzeDocumentFull returns structured response (requires GD)');
            return;
        }
        
        $service = new TextractService('test_key', 'test_secret');
        
        // Create minimal test image
        $image = imagecreate(50, 50);
        imagecolorallocate($image, 255, 255, 255);
        ob_start();
        imagejpeg($image);
        $imageData = ob_get_clean();
        imagedestroy($image);
        
        $result = $service->analyzeDocumentFull($imageData);
        
        $this->assert(
            is_array($result) && isset($result['success']),
            'analyzeDocumentFull returns structured response'
        );
    }
    
    private function testParseTablesEmptyResponse(): void
    {
        $service = new TextractService('test_key', 'test_secret');
        
        $emptyResponse = [];
        $tables = $service->parseTables($emptyResponse);
        
        $this->assert(
            is_array($tables) && count($tables) === 0,
            'parseTables handles empty response'
        );
    }
    
    private function testParseTablesWithData(): void
    {
        $service = new TextractService('test_key', 'test_secret');
        
        // Mock Textract response with table
        $mockResponse = [
            'Blocks' => [
                [
                    'Id' => 'table-1',
                    'BlockType' => 'TABLE',
                    'Confidence' => 99.5,
                    'Relationships' => [
                        [
                            'Type' => 'CHILD',
                            'Ids' => ['cell-1', 'cell-2']
                        ]
                    ]
                ],
                [
                    'Id' => 'cell-1',
                    'BlockType' => 'CELL',
                    'RowIndex' => 1,
                    'ColumnIndex' => 1,
                    'Confidence' => 99.0,
                    'Relationships' => [
                        [
                            'Type' => 'CHILD',
                            'Ids' => ['word-1']
                        ]
                    ]
                ],
                [
                    'Id' => 'cell-2',
                    'BlockType' => 'CELL',
                    'RowIndex' => 1,
                    'ColumnIndex' => 2,
                    'Confidence' => 98.5
                ],
                [
                    'Id' => 'word-1',
                    'BlockType' => 'WORD',
                    'Text' => 'Header'
                ]
            ]
        ];
        
        $tables = $service->parseTables($mockResponse);
        
        $this->assert(
            is_array($tables) && count($tables) === 1,
            'parseTables extracts table from response'
        );
        
        if (count($tables) > 0) {
            $table = $tables[0];
            $this->assert(
                isset($table['rows']) && isset($table['confidence']),
                'parseTables includes row data and confidence'
            );
        } else {
            $this->assert(false, 'parseTables includes row data and confidence');
        }
    }
    
    private function assert(bool $condition, string $message, string $error = ''): void
    {
        $this->results[] = [
            'passed' => $condition,
            'message' => $message,
            'error' => $error
        ];
        
        $status = $condition ? '✓' : '✗';
        $output = "  $status $message";
        if (!$condition && $error) {
            $output .= " - Error: $error";
        }
        echo "$output\n";
    }
    
    private function skip(string $message): void
    {
        $this->results[] = [
            'passed' => true,  // Count as passed so it doesn't fail
            'message' => $message,
            'skipped' => true
        ];
        echo "  ⊘ $message (skipped)\n";
    }
    
    private function displayResults(): void
    {
        $passed = count(array_filter($this->results, fn($r) => $r['passed']));
        $total = count($this->results);
        echo "\n";
    }
}

// Run tests if executed directly
if (basename(__FILE__) === basename($_SERVER['PHP_SELF'])) {
    $test = new TextractServiceTest();
    $test->run();
}
