<?php
/**
 * API v1: Get Thread Info
 * 
 * GET /api/v1/thread_info.php?thread_id=xxx&private_key=xxx
 * 
 * Response:
 * {
 *   "success": true,
 *   "thread": {
 *     "thread_id": "thread_abc123...",
 *     "created_at": "2025-12-18 14:30:00",
 *     "last_activity": "2025-12-18 15:45:00",
 *     "document_count": 3,
 *     "total_pii_found": 156,
 *     "total_api_calls": 45
 *   }
 * }
 */

require_once __DIR__ . '/../config.php';

use Redact\Classes\ThreadManager;

try {
    logRequest('thread_info');
    
    // Only allow GET
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        sendError('Method not allowed. Use GET', 405);
    }
    
    // Get parameters
    $threadId = $_GET['thread_id'] ?? null;
    $privateKey = $_GET['private_key'] ?? null;
    
    // Validate authentication
    validateThreadAuth($threadId, $privateKey);
    
    // Get thread info
    $threadManager = new ThreadManager();
    $thread = $threadManager->getThread($threadId);
    
    if (!$thread) {
        sendError('Thread not found', 404);
    }
    
    // Remove sensitive data
    unset($thread['metadata']['ip_address']);
    
    // Log success
    logRequest('thread_info', $threadId);
    
    // Send response
    sendResponse([
        'success' => true,
        'thread' => $thread
    ], 200);
    
} catch (Exception $e) {
    error_log('API Error (thread_info): ' . $e->getMessage());
    sendError('Internal server error', 500);
}

