<?php
/**
 * File Handler
 * Handles file uploads and conversions (PDF/Word to images)
 */

namespace Redact\Classes;

class FileHandler
{
    private int $maxFileSize;
    private array $allowedExtensions;
    
    public function __construct(int $maxFileSize = 5242880, array $allowedExtensions = ['pdf', 'jpg', 'jpeg', 'png'])
    {
        $this->maxFileSize = $maxFileSize; // Default 5MB
        $this->allowedExtensions = $allowedExtensions;
    }
    
    /**
     * Validate uploaded file
     *
     * @param array $file Uploaded file from $_FILES
     * @return array Result with success status and error message if any
     */
    public function validateFile(array $file): array
    {
        if (!isset($file['error']) || $file['error'] !== UPLOAD_ERR_OK) {
            return ['success' => false, 'error' => 'File upload failed'];
        }
        
        if ($file['size'] > $this->maxFileSize) {
            $maxMB = $this->maxFileSize / 1024 / 1024;
            return ['success' => false, 'error' => "File size exceeds {$maxMB}MB"];
        }
        
        $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if (!in_array($extension, $this->allowedExtensions)) {
            return ['success' => false, 'error' => 'Invalid file type'];
        }
        
        return ['success' => true];
    }
    
    /**
     * Convert file to images
     * Intelligently detects file type from content (MIME type), not just extension
     *
     * @param string $filePath Path to file
     * @param string $fileName Original file name (for logging/reference)
     * @return array Result with success status and image data array
     */
    public function convertToImages(string $filePath, string $fileName): array
    {
        // Detect actual file type from file content (more secure than extension)
        $mimeType = mime_content_type($filePath);
        
        // Normalize MIME type
        $mimeType = strtolower($mimeType);
        
        if ($mimeType === 'application/pdf') {
            // PDF file - requires Imagick conversion
            return $this->convertPDFToImages($filePath);
            
        } elseif (in_array($mimeType, ['image/jpeg', 'image/jpg', 'image/png', 'image/pjpeg'])) {
            // Already an image - no conversion needed
            $imageData = file_get_contents($filePath);
            
            if ($imageData === false) {
                return [
                    'success' => false,
                    'error' => 'Failed to read image file'
                ];
            }
            
            return [
                'success' => true,
                'images' => [$imageData],
                'detected_type' => $mimeType
            ];
            
        } else {
            // Unsupported file type
            $extension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
            return [
                'success' => false,
                'error' => "Unsupported file type. Detected MIME: {$mimeType}, Extension: {$extension}. Allowed: PDF, JPEG, PNG"
            ];
        }
    }
    
    /**
     * Convert PDF to images using Imagick
     *
     * @param string $pdfPath Path to PDF file
     * @return array Result with success status and images array
     */
    private function convertPDFToImages(string $pdfPath): array
    {
        if (!extension_loaded('imagick')) {
            return [
                'success' => false,
                'error' => 'Imagick extension is required for PDF processing'
            ];
        }
        
        try {
            $imagick = new \Imagick();
            $imagick->setResolution(300, 300);
            $imagick->readImage($pdfPath);
            
            $pageCount = $imagick->getNumberImages();
            $images = [];
            
            for ($i = 0; $i < $pageCount; $i++) {
                $imagick->setIteratorIndex($i);
                
                // Get only the current page (not all pages)
                $pageImage = $imagick->getImage();
                
                // Flatten layers to ensure all content is rendered (only this page)
                $flattenedImage = $pageImage->flattenImages();
                
                // Normalize to enhance contrast and visibility
                $flattenedImage->normalizeImage();
                
                $flattenedImage->setImageFormat('jpeg');
                $flattenedImage->setImageCompressionQuality(95);
                $images[] = $flattenedImage->getImageBlob();
                
                // Clean up
                $flattenedImage->clear();
                $flattenedImage->destroy();
                $pageImage->clear();
                $pageImage->destroy();
            }
            
            $imagick->clear();
            $imagick->destroy();
            
            return [
                'success' => true,
                'images' => $images,
                'page_count' => $pageCount,
                'detected_type' => 'application/pdf'
            ];
            
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => 'PDF conversion failed: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Convert Word document to images
     * Note: Requires additional setup (LibreOffice/unoconv or similar)
     *
     * @param string $docPath Path to Word document
     * @return array Result with success status and images array
     */
    public function convertWordToImages(string $docPath): array
    {
        // Convert Word to PDF first (requires LibreOffice or similar)
        // Then convert PDF to images
        
        return [
            'success' => false,
            'error' => 'Word document conversion not yet implemented'
        ];
    }
    
    /**
     * Get file info
     *
     * @param array $file Uploaded file from $_FILES
     * @return array File information
     */
    public function getFileInfo(array $file): array
    {
        $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        return [
            'name' => $file['name'],
            'size' => $file['size'],
            'type' => $file['type'],
            'extension' => $extension,
            'is_pdf' => $extension === 'pdf',
            'is_image' => in_array($extension, ['jpg', 'jpeg', 'png'])
        ];
    }
}

