<?php
/**
 * Unit Tests for FileHandler
 */

require_once __DIR__ . '/../src/classes/autoload.php';

use Redact\Classes\FileHandler;

class FileHandlerTest
{
    private array $results = [];
    private string $testDataDir;
    
    public function __construct()
    {
        $this->testDataDir = __DIR__ . '/test_data';
        if (!is_dir($this->testDataDir)) {
            mkdir($this->testDataDir, 0755, true);
        }
    }
    
    public function run(): void
    {
        echo "FileHandlerTest:\n";
        
        $this->testConstructor();
        $this->testConstructorCustomParams();
        $this->testValidateFileSuccess();
        $this->testValidateFileTooLarge();
        $this->testValidateFileInvalidType();
        $this->testGetFileInfo();
        $this->testConvertImageFile();
        $this->testMimeTypeDetection();
        $this->testPDFConversionSinglePage();
        $this->testPDFConversionMultiPage();
        $this->testPDFPagesAreSeparate();
        
        $this->displayResults();
    }
    
    private function testConstructor(): void
    {
        try {
            $handler = new FileHandler();
            $this->assert(true, 'Constructor with default parameters');
        } catch (Exception $e) {
            $this->assert(false, 'Constructor with default parameters', $e->getMessage());
        }
    }
    
    private function testConstructorCustomParams(): void
    {
        try {
            $handler = new FileHandler(10485760, ['pdf', 'jpg', 'png', 'docx']);
            $this->assert(true, 'Constructor with custom parameters');
        } catch (Exception $e) {
            $this->assert(false, 'Constructor with custom parameters', $e->getMessage());
        }
    }
    
    private function testValidateFileSuccess(): void
    {
        $handler = new FileHandler(5242880);
        
        $mockFile = [
            'name' => 'test.pdf',
            'type' => 'application/pdf',
            'size' => 1024000,
            'tmp_name' => '/tmp/test',
            'error' => UPLOAD_ERR_OK
        ];
        
        $result = $handler->validateFile($mockFile);
        
        $this->assert(
            $result['success'] === true,
            'validateFile accepts valid file'
        );
    }
    
    private function testValidateFileTooLarge(): void
    {
        $handler = new FileHandler(1048576); // 1MB
        
        $mockFile = [
            'name' => 'large.pdf',
            'type' => 'application/pdf',
            'size' => 5242880, // 5MB
            'tmp_name' => '/tmp/test',
            'error' => UPLOAD_ERR_OK
        ];
        
        $result = $handler->validateFile($mockFile);
        
        $this->assert(
            $result['success'] === false && strpos($result['error'], 'exceeds') !== false,
            'validateFile rejects file too large'
        );
    }
    
    private function testValidateFileInvalidType(): void
    {
        $handler = new FileHandler();
        
        $mockFile = [
            'name' => 'test.exe',
            'type' => 'application/exe',
            'size' => 1024,
            'tmp_name' => '/tmp/test',
            'error' => UPLOAD_ERR_OK
        ];
        
        $result = $handler->validateFile($mockFile);
        
        $this->assert(
            $result['success'] === false && strpos($result['error'], 'Invalid') !== false,
            'validateFile rejects invalid file type'
        );
    }
    
    private function testGetFileInfo(): void
    {
        $handler = new FileHandler();
        
        $mockFile = [
            'name' => 'document.pdf',
            'type' => 'application/pdf',
            'size' => 1024000,
            'tmp_name' => '/tmp/test',
            'error' => UPLOAD_ERR_OK
        ];
        
        $info = $handler->getFileInfo($mockFile);
        
        $this->assert(
            isset($info['name']) && isset($info['extension']) && $info['is_pdf'] === true,
            'getFileInfo returns complete file information'
        );
    }
    
    private function testConvertImageFile(): void
    {
        // Skip if GD is not available
        if (!extension_loaded('gd')) {
            $this->skip('convertToImages handles image file (requires GD extension)');
            return;
        }
        
        $handler = new FileHandler();
        
        // Create a simple test image
        $image = imagecreate(100, 100);
        $testFile = $this->testDataDir . '/test_image.jpg';
        imagecolorallocate($image, 255, 255, 255);
        imagejpeg($image, $testFile);
        imagedestroy($image);
        
        $result = $handler->convertToImages($testFile, 'test_image.jpg');
        
        $this->assert(
            $result['success'] === true && isset($result['images']) && count($result['images']) === 1,
            'convertToImages handles image file without conversion'
        );
        
        // Cleanup
        if (file_exists($testFile)) {
            unlink($testFile);
        }
    }
    
    private function testMimeTypeDetection(): void
    {
        // Skip if GD is not available
        if (!extension_loaded('gd')) {
            $this->skip('convertToImages detects MIME type (requires GD extension)');
            return;
        }
        
        $handler = new FileHandler();
        
        // Create test image
        $image = imagecreate(50, 50);
        $testFile = $this->testDataDir . '/mime_test.png';
        imagecolorallocate($image, 255, 255, 255);
        imagepng($image, $testFile);
        imagedestroy($image);
        
        $result = $handler->convertToImages($testFile, 'mime_test.png');
        
        $this->assert(
            $result['success'] === true && isset($result['detected_type']),
            'convertToImages detects MIME type from content'
        );
        
        // Cleanup
        if (file_exists($testFile)) {
            unlink($testFile);
        }
    }
    
    private function assert(bool $condition, string $message, string $error = ''): void
    {
        $this->results[] = [
            'passed' => $condition,
            'message' => $message,
            'error' => $error
        ];
        
        $status = $condition ? '✓' : '✗';
        $output = "  $status $message";
        if (!$condition && $error) {
            $output .= " - Error: $error";
        }
        echo "$output\n";
    }
    
    private function testPDFConversionSinglePage(): void
    {
        // Skip if Imagick is not available
        if (!extension_loaded('imagick')) {
            $this->skip('PDF conversion works for single page (requires Imagick)');
            return;
        }
        
        // Create a simple single-page PDF
        $testPdf = $this->createTestPDF(1);
        if (!$testPdf) {
            $this->skip('PDF conversion works for single page (PDF creation failed)');
            return;
        }
        
        $handler = new FileHandler();
        $result = $handler->convertToImages($testPdf, 'test.pdf');
        
        // Verify conversion succeeded
        $this->assert(
            $result['success'] === true,
            'PDF conversion succeeds for single-page PDF'
        );
        
        // Verify returns exactly 1 image
        $this->assert(
            isset($result['images']) && count($result['images']) === 1,
            'Single-page PDF returns exactly 1 image'
        );
        
        // Verify page count is correct
        $this->assert(
            isset($result['page_count']) && $result['page_count'] === 1,
            'Single-page PDF reports correct page_count'
        );
        
        // Cleanup
        if (file_exists($testPdf)) {
            unlink($testPdf);
        }
    }
    
    private function testPDFConversionMultiPage(): void
    {
        // Skip if Imagick is not available
        if (!extension_loaded('imagick')) {
            $this->skip('PDF conversion works for multi-page (requires Imagick)');
            return;
        }
        
        // Create a 3-page PDF
        $testPdf = $this->createTestPDF(3);
        if (!$testPdf) {
            $this->skip('PDF conversion works for multi-page (PDF creation failed)');
            return;
        }
        
        $handler = new FileHandler();
        $result = $handler->convertToImages($testPdf, 'test_multipage.pdf');
        
        // Verify conversion succeeded
        $this->assert(
            $result['success'] === true,
            'PDF conversion succeeds for multi-page PDF'
        );
        
        // Verify returns 3 separate images
        $this->assert(
            isset($result['images']) && count($result['images']) === 3,
            'Multi-page PDF returns separate image for each page (3 images)'
        );
        
        // Verify page count is correct
        $this->assert(
            isset($result['page_count']) && $result['page_count'] === 3,
            'Multi-page PDF reports correct page_count (3)'
        );
        
        // Cleanup
        if (file_exists($testPdf)) {
            unlink($testPdf);
        }
    }
    
    private function testPDFPagesAreSeparate(): void
    {
        // Skip if Imagick is not available
        if (!extension_loaded('imagick')) {
            $this->skip('PDF pages are converted separately, not combined (requires Imagick)');
            return;
        }
        
        // Create a 2-page PDF with different content on each page
        $testPdf = $this->createTestPDF(2);
        if (!$testPdf) {
            $this->skip('PDF pages are converted separately, not combined (PDF creation failed)');
            return;
        }
        
        $handler = new FileHandler();
        $result = $handler->convertToImages($testPdf, 'test_separate.pdf');
        
        if (!$result['success'] || !isset($result['images']) || count($result['images']) !== 2) {
            $this->assert(false, 'PDF pages are converted separately, not combined (conversion failed)');
            if (file_exists($testPdf)) unlink($testPdf);
            return;
        }
        
        $image1 = $result['images'][0];
        $image2 = $result['images'][1];
        
        // Verify each image is a valid JPEG blob
        $this->assert(
            strlen($image1) > 100 && strlen($image2) > 100,
            'Each page produces a valid image blob (not empty)'
        );
        
        // Verify images are different (not the same page duplicated)
        $this->assert(
            $image1 !== $image2,
            'Each page produces a DIFFERENT image (pages not combined/duplicated)'
        );
        
        // Verify we can load each image individually with Imagick
        try {
            $img1 = new \Imagick();
            $img1->readImageBlob($image1);
            $pageCount1 = $img1->getNumberImages();
            $img1->clear();
            
            $img2 = new \Imagick();
            $img2->readImageBlob($image2);
            $pageCount2 = $img2->getNumberImages();
            $img2->clear();
            
            $this->assert(
                $pageCount1 === 1 && $pageCount2 === 1,
                'Each converted image is a SINGLE page (not multi-page combined)'
            );
        } catch (Exception $e) {
            $this->assert(false, 'Each converted image is valid and loadable', $e->getMessage());
        }
        
        // Cleanup
        if (file_exists($testPdf)) {
            unlink($testPdf);
        }
    }
    
    /**
     * Create a test PDF with specified number of pages
     * Returns path to created PDF or false on failure
     */
    private function createTestPDF(int $pageCount): string|false
    {
        if (!extension_loaded('imagick')) {
            return false;
        }
        
        try {
            $pdf = new \Imagick();
            
            for ($i = 1; $i <= $pageCount; $i++) {
                // Create a unique image for each page
                $image = new \Imagick();
                $image->newImage(612, 792, new \ImagickPixel('white')); // Letter size
                $image->setImageFormat('pdf');
                
                // Add text to differentiate pages
                $draw = new \ImagickDraw();
                $draw->setFontSize(72);
                $draw->setFillColor(new \ImagickPixel('black'));
                $draw->annotation(200, 400, "Page $i");
                $image->drawImage($draw);
                
                $pdf->addImage($image);
                $image->clear();
                $image->destroy();
            }
            
            $testFile = $this->testDataDir . '/test_' . $pageCount . 'page_' . uniqid() . '.pdf';
            $pdf->setImageFormat('pdf');
            $pdf->writeImages($testFile, true);
            $pdf->clear();
            $pdf->destroy();
            
            return $testFile;
            
        } catch (Exception $e) {
            return false;
        }
    }
    
    private function skip(string $message): void
    {
        $this->results[] = [
            'passed' => true,  // Count as passed so it doesn't fail
            'message' => $message,
            'skipped' => true
        ];
        echo "  ⊘ $message (skipped)\n";
    }
    
    private function displayResults(): void
    {
        $passed = count(array_filter($this->results, fn($r) => $r['passed']));
        $total = count($this->results);
        echo "\n";
    }
    
    public function __destruct()
    {
        // Cleanup test data directory
        if (is_dir($this->testDataDir)) {
            $files = glob($this->testDataDir . '/*');
            foreach ($files as $file) {
                if (is_file($file)) {
                    unlink($file);
                }
            }
        }
    }
}

// Run tests if executed directly
if (basename(__FILE__) === basename($_SERVER['PHP_SELF'])) {
    $test = new FileHandlerTest();
    $test->run();
}
