<?php
/**
 * Page Count Verification Test
 * Traces through the entire flow to verify page count is correct at every step
 */

require_once __DIR__ . '/../src/classes/autoload.php';

use Redact\Classes\FileHandler;

class PageCountVerificationTest
{
    private array $results = [];
    private string $samplesDir;
    
    public function __construct()
    {
        $this->samplesDir = __DIR__ . '/Samples';
    }
    
    public function run(): void
    {
        echo "PageCountVerificationTest:\n";
        echo "Tracing page count through entire processing pipeline...\n\n";
        
        $this->testCompletePageCountFlow();
        
        $this->displayResults();
    }
    
    private function testCompletePageCountFlow(): void
    {
        $sampleFile = $this->samplesDir . '/UPDATED Regulated Employment Reference.pdf';
        
        if (!file_exists($sampleFile)) {
            $this->skip('Complete page count flow (file not found)');
            return;
        }
        
        if (!extension_loaded('imagick')) {
            $this->skip('Complete page count flow (requires Imagick)');
            return;
        }
        
        echo "  📄 Processing: UPDATED Regulated Employment Reference.pdf\n";
        echo "  " . str_repeat("─", 70) . "\n\n";
        
        // STEP 1: FileHandler conversion
        echo "  STEP 1: FileHandler.convertToImages()\n";
        $handler = new FileHandler();
        $conversionResult = $handler->convertToImages($sampleFile, 'UPDATED Regulated Employment Reference.pdf');
        
        if (!$conversionResult['success']) {
            $this->skip('Complete page count flow (conversion failed)');
            return;
        }
        
        $imagesCount = count($conversionResult['images'] ?? []);
        $reportedPageCount = $conversionResult['page_count'] ?? 0;
        
        echo "    ✓ Conversion successful\n";
        echo "    ✓ Images array: $imagesCount images\n";
        echo "    ✓ page_count field: $reportedPageCount\n";
        
        $this->assert(
            $imagesCount === $reportedPageCount,
            "FileHandler: images count matches page_count ($imagesCount = $reportedPageCount)"
        );
        
        // STEP 2: Simulate what PIIDetectionService does
        echo "\n  STEP 2: Building pages array (like PIIDetectionService)\n";
        
        $allPages = [];
        foreach ($conversionResult['images'] as $pageIndex => $imageData) {
            $pageNum = $pageIndex + 1;
            $allPages[] = [
                'page_number' => $pageNum,
                'image_data' => 'data:image/jpeg;base64,' . base64_encode($imageData)
            ];
        }
        
        $allPagesCount = count($allPages);
        echo "    ✓ Built pages array: $allPagesCount pages\n";
        
        $this->assert(
            $allPagesCount === $imagesCount,
            "Pages array length matches images count ($allPagesCount = $imagesCount)"
        );
        
        // STEP 3: Simulate API response (what PIIDetectionService returns)
        echo "\n  STEP 3: API response structure (PIIDetectionService.processDocument)\n";
        
        $apiResponse = [
            'success' => true,
            'total_pages' => count($allPages),  // Line 226 in PIIDetectionService
            'file_info' => [
                'page_count' => $reportedPageCount  // Line 113 in PIIDetectionService
            ],
            'pages' => $allPages
        ];
        
        echo "    ✓ total_pages: " . $apiResponse['total_pages'] . "\n";
        echo "    ✓ file_info.page_count: " . $apiResponse['file_info']['page_count'] . "\n";
        echo "    ✓ pages array length: " . count($apiResponse['pages']) . "\n";
        
        $this->assert(
            $apiResponse['total_pages'] === $reportedPageCount,
            "API total_pages matches FileHandler page_count ({$apiResponse['total_pages']} = $reportedPageCount)"
        );
        
        $this->assert(
            count($apiResponse['pages']) === $apiResponse['total_pages'],
            "API pages array length matches total_pages (" . count($apiResponse['pages']) . " = {$apiResponse['total_pages']})"
        );
        
        // STEP 4: Verify each page in array
        echo "\n  STEP 4: Verifying individual pages\n";
        
        $allPagesValid = true;
        $pageNumbers = [];
        
        foreach ($apiResponse['pages'] as $index => $page) {
            $expectedPageNum = $index + 1;
            $actualPageNum = $page['page_number'] ?? null;
            $pageNumbers[] = $actualPageNum;
            
            if ($actualPageNum !== $expectedPageNum) {
                $allPagesValid = false;
                echo "    ✗ Page $index: page_number is $actualPageNum (expected $expectedPageNum)\n";
            }
        }
        
        if ($allPagesValid) {
            echo "    ✓ All pages numbered correctly: [" . implode(', ', $pageNumbers) . "]\n";
        }
        
        $this->assert(
            $allPagesValid,
            "All pages have correct sequential page_number (1, 2, 3, ...)"
        );
        
        // FINAL VERIFICATION
        echo "\n  " . str_repeat("─", 70) . "\n";
        echo "  📊 FINAL VERIFICATION\n";
        echo "  " . str_repeat("─", 70) . "\n\n";
        
        $expected = 6;  // Known page count for the sample file
        
        echo "    Expected page count: $expected\n";
        echo "    FileHandler reported: $reportedPageCount\n";
        echo "    FileHandler images: $imagesCount\n";
        echo "    Service pages array: $allPagesCount\n";
        echo "    API total_pages: {$apiResponse['total_pages']}\n";
        echo "    API pages length: " . count($apiResponse['pages']) . "\n";
        
        $allMatch = (
            $reportedPageCount === $expected &&
            $imagesCount === $expected &&
            $allPagesCount === $expected &&
            $apiResponse['total_pages'] === $expected &&
            count($apiResponse['pages']) === $expected
        );
        
        echo "\n";
        $this->assert(
            $allMatch,
            "✅ ALL COUNTS MATCH: Expected=$expected, All steps return $expected pages"
        );
        
        if ($allMatch) {
            echo "\n  🎉 SUCCESS: Page count is CORRECT at every step in the pipeline!\n";
            echo "  ✓ FileHandler correctly detects 6 pages\n";
            echo "  ✓ FileHandler converts to 6 separate images\n";
            echo "  ✓ PIIDetectionService builds array with 6 pages\n";
            echo "  ✓ API returns total_pages: 6\n";
            echo "  ✓ API returns pages array with 6 entries\n";
        }
    }
    
    private function assert(bool $condition, string $message, string $error = ''): void
    {
        $this->results[] = [
            'passed' => $condition,
            'message' => $message,
            'error' => $error
        ];
        
        $status = $condition ? '✓' : '✗';
        $output = "  $status $message";
        if (!$condition && $error) {
            $output .= " - Error: $error";
        }
        echo "$output\n";
    }
    
    private function skip(string $message): void
    {
        $this->results[] = [
            'passed' => true,
            'message' => $message,
            'skipped' => true
        ];
        echo "  ⊘ $message (skipped)\n";
    }
    
    private function displayResults(): void
    {
        echo "\n";
        $passed = count(array_filter($this->results, fn($r) => $r['passed']));
        $failed = count(array_filter($this->results, fn($r) => !$r['passed']));
        $total = count($this->results);
        
        if ($failed === 0) {
            echo "  ═══════════════════════════════════════════════════════════════════\n";
            echo "  ✅ ALL VERIFICATIONS PASSED ($passed/$total)\n";
            echo "  ═══════════════════════════════════════════════════════════════════\n\n";
        } else {
            echo "  ❌ SOME VERIFICATIONS FAILED ($passed passed, $failed failed)\n\n";
        }
    }
}

// Run tests if executed directly
if (basename(__FILE__) === basename($_SERVER['PHP_SELF'])) {
    $test = new PageCountVerificationTest();
    $test->run();
}
