<?php
/**
 * Unit Tests for ThreadManager
 */

require_once __DIR__ . '/../src/classes/autoload.php';

use Redact\Classes\ThreadManager;

class ThreadManagerTest
{
    private array $results = [];
    private string $testDataDir;
    private ThreadManager $manager;
    
    public function __construct()
    {
        $this->testDataDir = __DIR__ . '/test_data/threads';
        if (!is_dir($this->testDataDir)) {
            mkdir($this->testDataDir, 0755, true);
        }
        
        $this->manager = new ThreadManager($this->testDataDir, 30);
    }
    
    public function run(): void
    {
        echo "ThreadManagerTest:\n";
        
        $this->testConstructor();
        $this->testCreateThread();
        $this->testCreateThreadWithMetadata();
        $this->testThreadExists();
        $this->testGetThread();
        $this->testUpdateThreadActivity();
        $this->testIncrementThreadStats();
        $this->testVerifyPrivateKey();
        $this->testListThreads();
        $this->testGetStatistics();
        $this->testDeleteThread();
        $this->testThreadExpiry();
        
        $this->displayResults();
    }
    
    private function testConstructor(): void
    {
        try {
            $manager = new ThreadManager($this->testDataDir);
            $this->assert(true, 'Constructor creates thread manager');
        } catch (Exception $e) {
            $this->assert(false, 'Constructor creates thread manager', $e->getMessage());
        }
    }
    
    private function testCreateThread(): void
    {
        $result = $this->manager->createThread();
        
        $this->assert(
            $result['success'] === true && isset($result['thread_id']) && isset($result['private_key']),
            'createThread returns thread_id and private_key'
        );
        
        // Cleanup
        if (isset($result['thread_id'])) {
            $this->manager->deleteThread($result['thread_id']);
        }
    }
    
    private function testCreateThreadWithMetadata(): void
    {
        $metadata = [
            'user_id' => 'test_user_123',
            'app_name' => 'TestApp',
            'description' => 'Test thread'
        ];
        
        $result = $this->manager->createThread($metadata);
        
        $this->assert(
            $result['success'] === true,
            'createThread accepts metadata'
        );
        
        if ($result['success']) {
            $thread = $this->manager->getThread($result['thread_id']);
            $this->assert(
                isset($thread['metadata']) && $thread['metadata']['user_id'] === 'test_user_123',
                'Thread stores metadata correctly'
            );
        } else {
            $this->assert(false, 'Thread stores metadata correctly');
        }
        
        // Cleanup
        if (isset($result['thread_id'])) {
            $this->manager->deleteThread($result['thread_id']);
        }
    }
    
    private function testThreadExists(): void
    {
        $result = $this->manager->createThread();
        $threadId = $result['thread_id'];
        
        $exists = $this->manager->threadExists($threadId);
        $this->assert($exists === true, 'threadExists returns true for existing thread');
        
        $notExists = $this->manager->threadExists('thread_nonexistent');
        $this->assert($notExists === false, 'threadExists returns false for non-existent thread');
        
        // Cleanup
        $this->manager->deleteThread($threadId);
    }
    
    private function testGetThread(): void
    {
        $result = $this->manager->createThread();
        $threadId = $result['thread_id'];
        
        $thread = $this->manager->getThread($threadId);
        
        $this->assert(
            is_array($thread) && isset($thread['thread_id']) && $thread['thread_id'] === $threadId,
            'getThread returns thread data'
        );
        
        // Cleanup
        $this->manager->deleteThread($threadId);
    }
    
    private function testUpdateThreadActivity(): void
    {
        $result = $this->manager->createThread();
        $threadId = $result['thread_id'];
        
        sleep(1); // Wait 1 second
        
        $updated = $this->manager->updateThreadActivity($threadId);
        
        $this->assert(
            $updated === true,
            'updateThreadActivity updates last_activity timestamp'
        );
        
        // Cleanup
        $this->manager->deleteThread($threadId);
    }
    
    private function testIncrementThreadStats(): void
    {
        $result = $this->manager->createThread();
        $threadId = $result['thread_id'];
        
        $incremented = $this->manager->incrementThreadStats($threadId, [
            'document_count' => 1,
            'total_pii_found' => 25,
            'total_api_calls' => 5
        ]);
        
        $this->assert($incremented === true, 'incrementThreadStats updates statistics');
        
        $thread = $this->manager->getThread($threadId);
        $this->assert(
            $thread['document_count'] === 1 && $thread['total_pii_found'] === 25,
            'Thread statistics are correctly incremented'
        );
        
        // Cleanup
        $this->manager->deleteThread($threadId);
    }
    
    private function testVerifyPrivateKey(): void
    {
        $result = $this->manager->createThread();
        $threadId = $result['thread_id'];
        $privateKey = $result['private_key'];
        
        $valid = $this->manager->verifyPrivateKey($threadId, $privateKey);
        $this->assert($valid === true, 'verifyPrivateKey accepts correct key');
        
        $invalid = $this->manager->verifyPrivateKey($threadId, 'wrong_key');
        $this->assert($invalid === false, 'verifyPrivateKey rejects incorrect key');
        
        // Cleanup
        $this->manager->deleteThread($threadId);
    }
    
    private function testListThreads(): void
    {
        // Create 2 threads
        $result1 = $this->manager->createThread();
        $result2 = $this->manager->createThread();
        
        $threads = $this->manager->listThreads();
        
        $this->assert(
            is_array($threads) && count($threads) >= 2,
            'listThreads returns array of threads'
        );
        
        // Cleanup
        $this->manager->deleteThread($result1['thread_id']);
        $this->manager->deleteThread($result2['thread_id']);
    }
    
    private function testGetStatistics(): void
    {
        $result = $this->manager->createThread();
        $threadId = $result['thread_id'];
        
        $stats = $this->manager->getStatistics();
        
        $this->assert(
            is_array($stats) && isset($stats['total_threads']) && $stats['total_threads'] >= 1,
            'getStatistics returns comprehensive statistics'
        );
        
        // Cleanup
        $this->manager->deleteThread($threadId);
    }
    
    private function testDeleteThread(): void
    {
        $result = $this->manager->createThread();
        $threadId = $result['thread_id'];
        $privateKey = $result['private_key'];
        
        $deleted = $this->manager->deleteThread($threadId, $privateKey);
        
        $this->assert(
            $deleted['success'] === true,
            'deleteThread removes thread and all data'
        );
        
        $exists = $this->manager->threadExists($threadId);
        $this->assert($exists === false, 'Deleted thread no longer exists');
    }
    
    private function testThreadExpiry(): void
    {
        // Create manager with 0-day expiry (immediate expiry)
        $shortManager = new ThreadManager($this->testDataDir . '_expiry', 0);
        
        $result = $shortManager->createThread();
        $threadId = $result['thread_id'];
        
        // Thread should be expired immediately
        $isValid = $shortManager->isThreadValid($threadId);
        
        // Wait a moment and check cleanup
        sleep(1);
        $cleanupResult = $shortManager->cleanupExpiredThreads();
        
        $this->assert(
            is_array($cleanupResult) && $cleanupResult['success'] === true,
            'cleanupExpiredThreads removes old threads'
        );
    }
    
    private function assert(bool $condition, string $message, string $error = ''): void
    {
        $this->results[] = [
            'passed' => $condition,
            'message' => $message,
            'error' => $error
        ];
        
        $status = $condition ? '✓' : '✗';
        $output = "  $status $message";
        if (!$condition && $error) {
            $output .= " - Error: $error";
        }
        echo "$output\n";
    }
    
    private function displayResults(): void
    {
        $passed = count(array_filter($this->results, fn($r) => $r['passed']));
        $total = count($this->results);
        echo "\n";
    }
    
    public function __destruct()
    {
        // Cleanup test data
        $this->cleanupDirectory($this->testDataDir);
        $this->cleanupDirectory($this->testDataDir . '_expiry');
    }
    
    private function cleanupDirectory(string $dir): void
    {
        if (!is_dir($dir)) {
            return;
        }
        
        $items = array_diff(scandir($dir), ['.', '..']);
        foreach ($items as $item) {
            $path = $dir . '/' . $item;
            if (is_dir($path)) {
                $this->cleanupDirectory($path);
            } else {
                unlink($path);
            }
        }
        rmdir($dir);
    }
}

// Run tests if executed directly
if (basename(__FILE__) === basename($_SERVER['PHP_SELF'])) {
    $test = new ThreadManagerTest();
    $test->run();
}
