<?php
/**
 * Unit Tests for FileHandler
 */

use Redact\Classes\FileHandler;

class FileHandlerTest extends BaseTest
{
    private string $testPdfPath;
    
    public function runTests(): void
    {
        $this->testPdfPath = __DIR__ . '/BeytekinS Payslips.pdf';
        
        $this->testFileValidation();
        $this->testFileInfo();
        $this->testPDFConversion();
        $this->testImageValidation();
        $this->testInvalidFileValidation();
    }
    
    private function testFileValidation(): void
    {
        $fileHandler = new FileHandler(5 * 1024 * 1024); // 5MB
        
        // Simulate valid file upload
        $validFile = [
            'name' => 'test.pdf',
            'tmp_name' => $this->testPdfPath,
            'size' => filesize($this->testPdfPath),
            'type' => 'application/pdf',
            'error' => UPLOAD_ERR_OK
        ];
        
        $result = $fileHandler->validateFile($validFile);
        
        $this->runner->assertTrue($result['success'], 'Valid PDF file should pass validation');
    }
    
    private function testFileInfo(): void
    {
        $fileHandler = new FileHandler();
        
        $file = [
            'name' => 'BeytekinS Payslips.pdf',
            'tmp_name' => $this->testPdfPath,
            'size' => filesize($this->testPdfPath),
            'type' => 'application/pdf',
            'error' => UPLOAD_ERR_OK
        ];
        
        $info = $fileHandler->getFileInfo($file);
        
        $this->runner->assertEquals('pdf', $info['extension'], 'File extension should be pdf');
        $this->runner->assertTrue($info['is_pdf'], 'Should recognize file as PDF');
        $this->runner->assertFalse($info['is_image'], 'Should not recognize PDF as image');
    }
    
    private function testPDFConversion(): void
    {
        if (!extension_loaded('imagick')) {
            echo "  ⏭️  Skipping PDF conversion test (Imagick not installed)\n";
            return;
        }
        
        $fileHandler = new FileHandler();
        
        $result = $fileHandler->convertToImages($this->testPdfPath, 'test.pdf');
        
        $this->runner->assertTrue($result['success'], 'PDF conversion should succeed');
        $this->runner->assertNotNull($result['images'] ?? null, 'Should return images array');
        $this->runner->assertGreaterThan(0, count($result['images'] ?? []), 'Should have at least one image');
        
        // Check if images are valid
        if (isset($result['images'][0])) {
            $this->runner->assertGreaterThan(1000, strlen($result['images'][0]), 'Image data should be substantial');
        }
    }
    
    private function testImageValidation(): void
    {
        $fileHandler = new FileHandler();
        
        // Simulate valid image upload
        $validImage = [
            'name' => 'test.jpg',
            'tmp_name' => '/tmp/test.jpg',
            'size' => 1024 * 500, // 500KB
            'type' => 'image/jpeg',
            'error' => UPLOAD_ERR_OK
        ];
        
        $result = $fileHandler->validateFile($validImage);
        
        $this->runner->assertTrue($result['success'], 'Valid image file should pass validation');
    }
    
    private function testInvalidFileValidation(): void
    {
        $fileHandler = new FileHandler(1024); // 1KB max
        
        // File too large
        $largeFile = [
            'name' => 'large.pdf',
            'tmp_name' => $this->testPdfPath,
            'size' => 10 * 1024 * 1024, // 10MB
            'type' => 'application/pdf',
            'error' => UPLOAD_ERR_OK
        ];
        
        $result = $fileHandler->validateFile($largeFile);
        
        $this->runner->assertFalse($result['success'], 'Large file should fail validation');
        
        // Invalid extension
        $invalidFile = [
            'name' => 'test.exe',
            'tmp_name' => '/tmp/test.exe',
            'size' => 1024,
            'type' => 'application/x-executable',
            'error' => UPLOAD_ERR_OK
        ];
        
        $result = $fileHandler->validateFile($invalidFile);
        
        $this->runner->assertFalse($result['success'], 'Invalid file type should fail validation');
    }
}

