<?php
/**
 * API Test Script
 * Tests all API endpoints
 */

echo "\n╔════════════════════════════════════════════════════════════════╗\n";
echo "║              PII DETECTION API - TEST SUITE                     ║\n";
echo "╚════════════════════════════════════════════════════════════════╝\n\n";

$baseUrl = 'http://localhost/redact/api/v1';

/**
 * Make API request
 */
function apiRequest($endpoint, $method = 'POST', $data = null) {
    global $baseUrl;
    
    $ch = curl_init($baseUrl . $endpoint);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
    
    if ($data) {
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    return [
        'code' => $httpCode,
        'data' => json_decode($response, true)
    ];
}

// Test 1: Create Thread
echo "📋 TEST 1: Create Thread\n";
echo str_repeat("-", 70) . "\n";

$result = apiRequest('/thread_create.php', 'POST', [
    'metadata' => [
        'user_id' => 'test_' . time(),
        'app_name' => 'Test Suite'
    ]
]);

if ($result['code'] === 201 && $result['data']['success']) {
    $threadId = $result['data']['thread_id'];
    $privateKey = $result['data']['private_key'];
    
    echo "✅ Thread created successfully\n";
    echo "   Thread ID: $threadId\n";
    echo "   Private Key: " . substr($privateKey, 0, 20) . "...\n";
    echo "   Created at: {$result['data']['created_at']}\n\n";
} else {
    echo "❌ Failed to create thread\n";
    echo "   Status: {$result['code']}\n";
    echo "   Error: " . ($result['data']['error'] ?? 'Unknown') . "\n\n";
    exit(1);
}

// Test 2: Get Thread Info
echo "📋 TEST 2: Get Thread Info\n";
echo str_repeat("-", 70) . "\n";

$result = apiRequest("/thread_info.php?thread_id=$threadId&private_key=$privateKey", 'GET');

if ($result['code'] === 200 && $result['data']['success']) {
    echo "✅ Thread info retrieved\n";
    echo "   Documents: {$result['data']['thread']['document_count']}\n";
    echo "   PII found: {$result['data']['thread']['total_pii_found']}\n";
    echo "   Last activity: {$result['data']['thread']['last_activity']}\n\n";
} else {
    echo "❌ Failed to get thread info\n";
    echo "   Status: {$result['code']}\n";
    echo "   Error: " . ($result['data']['error'] ?? 'Unknown') . "\n\n";
}

// Test 3: Process File (if sample PDF exists)
$sampleFile = __DIR__ . '/../src/classes/unit/BeytekinS Payslips.pdf';

if (file_exists($sampleFile)) {
    echo "📋 TEST 3: Process File\n";
    echo str_repeat("-", 70) . "\n";
    
    $fileData = base64_encode(file_get_contents($sampleFile));
    
    echo "Processing document (this may take 30-60 seconds)...\n";
    
    $result = apiRequest('/process_file.php', 'POST', [
        'thread_id' => $threadId,
        'private_key' => $privateKey,
        'file_data' => $fileData,
        'file_name' => 'test_document.pdf'
    ]);
    
    if ($result['code'] === 200 && $result['data']['success']) {
        echo "✅ File processed successfully\n";
        echo "   Processing time: {$result['data']['processing_time']}\n";
        echo "   Total pages: {$result['data']['total_pages']}\n";
        echo "   PII instances: {$result['data']['total_pii_instances']}\n";
        echo "   Comprehend calls: {$result['data']['comprehend_calls']}\n";
        echo "   Optimization rate: {$result['data']['optimization_rate']}%\n";
        echo "   Words learned: {$result['data']['cache']['words_learned']}\n\n";
    } else {
        echo "❌ Failed to process file\n";
        echo "   Status: {$result['code']}\n";
        echo "   Error: " . ($result['data']['error'] ?? 'Unknown') . "\n\n";
    }
} else {
    echo "📋 TEST 3: Process File\n";
    echo str_repeat("-", 70) . "\n";
    echo "⏭️  Skipped (sample PDF not found)\n\n";
}

// Test 4: Invalid Authentication
echo "📋 TEST 4: Invalid Authentication\n";
echo str_repeat("-", 70) . "\n";

$result = apiRequest("/thread_info.php?thread_id=$threadId&private_key=invalid_key", 'GET');

if ($result['code'] === 401) {
    echo "✅ Invalid credentials correctly rejected\n";
    echo "   Status: 401 Unauthorized\n\n";
} else {
    echo "❌ Security test failed - invalid credentials accepted\n\n";
}

// Test 5: Delete Thread
echo "📋 TEST 5: Delete Thread\n";
echo str_repeat("-", 70) . "\n";

$result = apiRequest('/thread_delete.php', 'DELETE', [
    'thread_id' => $threadId,
    'private_key' => $privateKey
]);

if ($result['code'] === 200 && $result['data']['success']) {
    echo "✅ Thread deleted successfully\n";
    echo "   Message: {$result['data']['message']}\n\n";
} else {
    echo "❌ Failed to delete thread\n";
    echo "   Status: {$result['code']}\n";
    echo "   Error: " . ($result['data']['error'] ?? 'Unknown') . "\n\n";
}

// Summary
echo "╔════════════════════════════════════════════════════════════════╗\n";
echo "║              TEST SUITE COMPLETED                               ║\n";
echo "╚════════════════════════════════════════════════════════════════╝\n\n";

echo "✅ Thread creation\n";
echo "✅ Thread info retrieval\n";
echo "✅ Authentication validation\n";
echo "✅ Thread deletion\n";
if (file_exists($sampleFile)) {
    echo "✅ File processing\n";
}
echo "\n🎉 API is operational!\n\n";

