<?php
/**
 * Simple Test Runner
 * Runs all unit tests and reports results
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

// Load classes
require_once __DIR__ . '/../autoload.php';
require_once __DIR__ . '/../../config/config.php';

class TestRunner
{
    private int $totalTests = 0;
    private int $passedTests = 0;
    private int $failedTests = 0;
    private array $failures = [];
    
    public function runAllTests(): void
    {
        echo "\n╔════════════════════════════════════════════════════════════════╗\n";
        echo "║              PII DETECTION SERVICE - UNIT TESTS                 ║\n";
        echo "╚════════════════════════════════════════════════════════════════╝\n\n";
        
        $startTime = microtime(true);
        
        // Run all test files
        $this->runTestFile('FileHandlerTest.php');
        $this->runTestFile('TextractServiceTest.php');
        $this->runTestFile('ComprehendServiceTest.php');
        $this->runTestFile('RegistryManagerTest.php');
        $this->runTestFile('PIIDetectionServiceTest.php');
        
        $endTime = microtime(true);
        $duration = round($endTime - $startTime, 2);
        
        // Print summary
        echo "\n" . str_repeat("=", 70) . "\n";
        echo "SUMMARY\n";
        echo str_repeat("=", 70) . "\n";
        echo "Total Tests:  {$this->totalTests}\n";
        echo "✅ Passed:     {$this->passedTests}\n";
        echo "❌ Failed:     {$this->failedTests}\n";
        echo "Duration:     {$duration}s\n";
        
        if ($this->failedTests > 0) {
            echo "\n❌ FAILURES:\n";
            foreach ($this->failures as $failure) {
                echo "  - {$failure}\n";
            }
            exit(1);
        } else {
            echo "\n🎉 ALL TESTS PASSED!\n";
            exit(0);
        }
    }
    
    private function runTestFile(string $filename): void
    {
        $filepath = __DIR__ . '/' . $filename;
        if (!file_exists($filepath)) {
            echo "⚠️  Test file not found: $filename\n";
            return;
        }
        
        require_once $filepath;
        
        // Extract class name from filename
        $className = str_replace('.php', '', $filename);
        
        if (!class_exists($className)) {
            echo "⚠️  Test class not found: $className\n";
            return;
        }
        
        $test = new $className();
        $test->setRunner($this);
        
        echo "\n📋 Running: $className\n";
        echo str_repeat("-", 70) . "\n";
        
        $test->runTests();
    }
    
    public function assert(bool $condition, string $testName): void
    {
        $this->totalTests++;
        
        if ($condition) {
            $this->passedTests++;
            echo "  ✅ $testName\n";
        } else {
            $this->failedTests++;
            $this->failures[] = $testName;
            echo "  ❌ $testName\n";
        }
    }
    
    public function assertEquals($expected, $actual, string $testName): void
    {
        $this->assert($expected === $actual, $testName . " (expected: " . json_encode($expected) . ", got: " . json_encode($actual) . ")");
    }
    
    public function assertTrue($value, string $testName): void
    {
        $this->assert($value === true, $testName);
    }
    
    public function assertFalse($value, string $testName): void
    {
        $this->assert($value === false, $testName);
    }
    
    public function assertNotNull($value, string $testName): void
    {
        $this->assert($value !== null, $testName);
    }
    
    public function assertGreaterThan($threshold, $value, string $testName): void
    {
        $this->assert($value > $threshold, $testName . " (expected > $threshold, got: $value)");
    }
}

abstract class BaseTest
{
    protected ?TestRunner $runner = null;
    
    public function setRunner(TestRunner $runner): void
    {
        $this->runner = $runner;
    }
    
    abstract public function runTests(): void;
}

// Run all tests if executed directly
if (basename(__FILE__) === basename($_SERVER['PHP_SELF'])) {
    $runner = new TestRunner();
    $runner->runAllTests();
}

