<?php
/**
 * Unit Tests for RegistryManager
 */

use Redact\Classes\RegistryManager;
use Redact\Classes\TextractService;
use Redact\Classes\FileHandler;

class RegistryManagerTest extends BaseTest
{
    private array $sampleBlocks;
    
    public function runTests(): void
    {
        $this->setupSampleBlocks();
        
        $this->testRegistryInitialization();
        $this->testBuildRegistries();
        $this->testLayoutExtraction();
        $this->testWordRegistryBuilding();
        $this->testLayoutsToProcess();
        $this->testPIIMapping();
        $this->testApplyPIIToPage();
        $this->testStatistics();
    }
    
    private function setupSampleBlocks(): void
    {
        // Create mock Textract blocks
        $this->sampleBlocks = [
            [
                'Id' => 'layout-1',
                'BlockType' => 'LAYOUT_TABLE',
                'Confidence' => 95.5,
                'Geometry' => [
                    'BoundingBox' => [
                        'Left' => 0.1,
                        'Top' => 0.2,
                        'Width' => 0.8,
                        'Height' => 0.1
                    ]
                ],
                'Relationships' => [
                    [
                        'Type' => 'CHILD',
                        'Ids' => ['line-1']
                    ]
                ]
            ],
            [
                'Id' => 'line-1',
                'BlockType' => 'LINE',
                'Text' => 'John Smith',
                'Relationships' => [
                    [
                        'Type' => 'CHILD',
                        'Ids' => ['word-1', 'word-2']
                    ]
                ]
            ],
            [
                'Id' => 'word-1',
                'BlockType' => 'WORD',
                'Text' => 'John',
                'Confidence' => 99.5,
                'Geometry' => [
                    'BoundingBox' => [
                        'Left' => 0.1,
                        'Top' => 0.2,
                        'Width' => 0.05,
                        'Height' => 0.02
                    ]
                ]
            ],
            [
                'Id' => 'word-2',
                'BlockType' => 'WORD',
                'Text' => 'Smith',
                'Confidence' => 99.3,
                'Geometry' => [
                    'BoundingBox' => [
                        'Left' => 0.16,
                        'Top' => 0.2,
                        'Width' => 0.05,
                        'Height' => 0.02
                    ]
                ]
            ]
        ];
    }
    
    private function testRegistryInitialization(): void
    {
        $registry = new RegistryManager();
        
        $this->runner->assertNotNull($registry, 'RegistryManager should initialize');
    }
    
    private function testBuildRegistries(): void
    {
        $registry = new RegistryManager();
        
        [$layouts, $words] = $registry->buildRegistriesFromBlocks($this->sampleBlocks, 1);
        
        $this->runner->assertNotNull($layouts, 'Should return layouts array');
        $this->runner->assertNotNull($words, 'Should return words array');
        $this->runner->assertGreaterThan(0, count($layouts), 'Should extract layouts');
        $this->runner->assertGreaterThan(0, count($words), 'Should extract words');
    }
    
    private function testLayoutExtraction(): void
    {
        $registry = new RegistryManager();
        
        [$layouts, $words] = $registry->buildRegistriesFromBlocks($this->sampleBlocks, 1);
        
        if (count($layouts) > 0) {
            $layout = $layouts[0];
            
            $this->runner->assertEquals('page1_layout0', $layout['id'], 'Layout should have correct ID');
            $this->runner->assertEquals('LAYOUT_TABLE', $layout['type'], 'Layout should have correct type');
            $this->runner->assertEquals(1, $layout['page'], 'Layout should have correct page number');
            $this->runner->assertNotNull($layout['word_blocks'] ?? null, 'Layout should have word_blocks');
            $this->runner->assertNotNull($layout['text'] ?? null, 'Layout should have text');
        }
    }
    
    private function testWordRegistryBuilding(): void
    {
        $registry = new RegistryManager();
        
        [$layouts, $words] = $registry->buildRegistriesFromBlocks($this->sampleBlocks, 1);
        
        // Build registry from multiple pages to test word tracking
        [$layouts2, $words2] = $registry->buildRegistriesFromBlocks($this->sampleBlocks, 2);
        
        $stats = $registry->getStatistics();
        
        $this->runner->assertGreaterThan(0, $stats['unique_words'], 'Should track unique words');
        $this->runner->assertGreaterThan(0, $stats['total_words'], 'Should track total word occurrences');
    }
    
    private function testLayoutsToProcess(): void
    {
        $registry = new RegistryManager();
        
        [$layouts, $words] = $registry->buildRegistriesFromBlocks($this->sampleBlocks, 1);
        
        $layoutsToProcess = $registry->getLayoutsToProcess();
        
        $this->runner->assertNotNull($layoutsToProcess, 'Should return layouts to process array');
        
        // All layouts should need processing initially
        $this->runner->assertGreaterThan(0, count($layoutsToProcess), 'Should have layouts to process');
    }
    
    private function testPIIMapping(): void
    {
        $registry = new RegistryManager();
        
        [$layouts, $words] = $registry->buildRegistriesFromBlocks($this->sampleBlocks, 1);
        
        // Mock PII entities
        $mockEntities = [
            [
                'Type' => 'NAME',
                'Score' => 0.99,
                'BeginOffset' => 0,
                'EndOffset' => 4 // "John"
            ]
        ];
        
        $layoutId = 'page1_layout0';
        $layoutText = 'John Smith';
        
        $registry->mapPIIToWordBlocks($mockEntities, $layoutId, $layoutText);
        
        $stats = $registry->getStatistics();
        
        $this->runner->assertGreaterThan(0, $stats['pii_words'], 'Should have PII words after mapping');
    }
    
    private function testApplyPIIToPage(): void
    {
        $registry = new RegistryManager();
        
        [$layouts, $words] = $registry->buildRegistriesFromBlocks($this->sampleBlocks, 1);
        
        // Mock PII mapping
        $mockEntities = [
            [
                'Type' => 'NAME',
                'Score' => 0.99,
                'BeginOffset' => 0,
                'EndOffset' => 4
            ]
        ];
        
        $registry->mapPIIToWordBlocks($mockEntities, 'page1_layout0', 'John Smith');
        
        $pageData = [
            'word_blocks' => $words
        ];
        
        $piiBlocks = $registry->applyPIIToPage($pageData);
        
        $this->runner->assertNotNull($piiBlocks, 'Should return PII blocks array');
        
        if (count($piiBlocks) > 0) {
            $this->runner->assertGreaterThan(0, count($piiBlocks), 'Should have PII blocks');
            
            $piiBlock = $piiBlocks[0];
            $this->runner->assertNotNull($piiBlock['Type'] ?? null, 'PII block should have Type');
            $this->runner->assertNotNull($piiBlock['Text'] ?? null, 'PII block should have Text');
            $this->runner->assertNotNull($piiBlock['Score'] ?? null, 'PII block should have Score');
        }
    }
    
    private function testStatistics(): void
    {
        $registry = new RegistryManager();
        
        [$layouts, $words] = $registry->buildRegistriesFromBlocks($this->sampleBlocks, 1);
        
        $stats = $registry->getStatistics();
        
        $this->runner->assertNotNull($stats['total_layouts'] ?? null, 'Stats should include total_layouts');
        $this->runner->assertNotNull($stats['unique_words'] ?? null, 'Stats should include unique_words');
        $this->runner->assertNotNull($stats['total_words'] ?? null, 'Stats should include total_words');
        $this->runner->assertNotNull($stats['pii_words'] ?? null, 'Stats should include pii_words');
    }
}

