<?php
/**
 * Integration Tests for PIIDetectionService
 * Tests the complete workflow using all classes together
 */

use Redact\Classes\PIIDetectionService;

class PIIDetectionServiceTest extends BaseTest
{
    private string $testPdfPath;
    private array $awsCredentials;
    
    public function runTests(): void
    {
        $this->testPdfPath = __DIR__ . '/BeytekinS Payslips.pdf';
        $this->awsCredentials = getConfig('AWS_Credentials');
        
        if (!$this->awsCredentials || !isset($this->awsCredentials['key'])) {
            echo "  ⏭️  Skipping PIIDetectionService tests (AWS credentials not configured)\n";
            return;
        }
        
        if (!extension_loaded('imagick')) {
            echo "  ⏭️  Skipping PIIDetectionService tests (Imagick not installed)\n";
            return;
        }
        
        $this->testServiceInitialization();
        $this->testProcessDocumentFromPath();
        $this->testProcessDocumentUpload();
        $this->testResultStructure();
        $this->testPIIDetectionInDocument();
    }
    
    private function testServiceInitialization(): void
    {
        $service = new PIIDetectionService($this->awsCredentials);
        
        $this->runner->assertNotNull($service, 'PIIDetectionService should initialize');
    }
    
    private function testProcessDocumentFromPath(): void
    {
        if (!file_exists($this->testPdfPath)) {
            echo "  ⏭️  Skipping test (sample PDF not found)\n";
            return;
        }
        
        $service = new PIIDetectionService($this->awsCredentials, [
            'region' => 'us-east-1',
            'max_file_size' => 10 * 1024 * 1024
        ]);
        
        echo "\n  🔄 Processing document (this may take 30-60 seconds)...\n";
        
        $result = $service->processDocument($this->testPdfPath);
        
        $this->runner->assertTrue($result['success'], 'Document processing should succeed');
        $this->runner->assertNotNull($result['pages'] ?? null, 'Result should include pages');
    }
    
    private function testProcessDocumentUpload(): void
    {
        if (!file_exists($this->testPdfPath)) {
            echo "  ⏭️  Skipping test (sample PDF not found)\n";
            return;
        }
        
        $service = new PIIDetectionService($this->awsCredentials);
        
        // Simulate uploaded file
        $uploadedFile = [
            'tmp_name' => $this->testPdfPath,
            'name' => 'BeytekinS Payslips.pdf',
            'size' => filesize($this->testPdfPath),
            'type' => 'application/pdf',
            'error' => UPLOAD_ERR_OK
        ];
        
        echo "\n  🔄 Processing uploaded document (this may take 30-60 seconds)...\n";
        
        $result = $service->processUploadedFile($uploadedFile);
        
        $this->runner->assertTrue($result['success'], 'Upload processing should succeed');
    }
    
    private function testResultStructure(): void
    {
        if (!file_exists($this->testPdfPath)) {
            echo "  ⏭️  Skipping test (sample PDF not found)\n";
            return;
        }
        
        $service = new PIIDetectionService($this->awsCredentials);
        
        $result = $service->processDocument($this->testPdfPath);
        
        if ($result['success']) {
            // Check result structure
            $this->runner->assertNotNull($result['processing_time'] ?? null, 'Should include processing_time');
            $this->runner->assertNotNull($result['total_pages'] ?? null, 'Should include total_pages');
            $this->runner->assertNotNull($result['layout_count'] ?? null, 'Should include layout_count');
            $this->runner->assertNotNull($result['unique_words'] ?? null, 'Should include unique_words');
            $this->runner->assertNotNull($result['pii_words'] ?? null, 'Should include pii_words');
            $this->runner->assertNotNull($result['total_pii_instances'] ?? null, 'Should include total_pii_instances');
            $this->runner->assertNotNull($result['comprehend_calls'] ?? null, 'Should include comprehend_calls');
            $this->runner->assertNotNull($result['pages'] ?? null, 'Should include pages array');
            
            // Check page structure
            if (count($result['pages']) > 0) {
                $page = $result['pages'][0];
                
                $this->runner->assertNotNull($page['page_number'] ?? null, 'Page should have page_number');
                $this->runner->assertNotNull($page['layouts'] ?? null, 'Page should have layouts');
                $this->runner->assertNotNull($page['word_blocks'] ?? null, 'Page should have word_blocks');
                $this->runner->assertNotNull($page['pii_blocks'] ?? null, 'Page should have pii_blocks');
                $this->runner->assertNotNull($page['image_data'] ?? null, 'Page should have image_data');
            }
        }
    }
    
    private function testPIIDetectionInDocument(): void
    {
        if (!file_exists($this->testPdfPath)) {
            echo "  ⏭️  Skipping test (sample PDF not found)\n";
            return;
        }
        
        $service = new PIIDetectionService($this->awsCredentials);
        
        $result = $service->processDocument($this->testPdfPath);
        
        if ($result['success']) {
            $this->runner->assertGreaterThan(0, $result['total_pages'], 'Should process at least one page');
            $this->runner->assertGreaterThan(0, $result['layout_count'], 'Should detect layouts');
            $this->runner->assertGreaterThan(0, $result['unique_words'], 'Should extract words');
            
            // Check if PII was detected
            if ($result['total_pii_instances'] > 0) {
                $this->runner->assertGreaterThan(0, $result['pii_words'], 'Should identify PII words');
                $this->runner->assertGreaterThan(0, $result['total_pii_instances'], 'Should find PII instances');
                
                echo "\n  📊 PII Detection Results:\n";
                echo "     - Unique words: {$result['unique_words']}\n";
                echo "     - PII words: {$result['pii_words']}\n";
                echo "     - Total PII instances: {$result['total_pii_instances']}\n";
                echo "     - Comprehend calls: {$result['comprehend_calls']}\n";
                echo "     - Optimization: {$result['optimization_rate']}%\n";
            } else {
                echo "\n  ℹ️  No PII detected in document (this may vary by content)\n";
            }
            
            // Check comprehend efficiency
            if ($result['layout_count'] > 0) {
                $efficiency = ($result['layouts_skipped'] / $result['layout_count']) * 100;
                if ($efficiency > 0) {
                    echo "  ⚡ Optimization achieved: {$efficiency}% layouts skipped\n";
                }
            }
        }
    }
}

